/*
	$DOCFILE:CAPIDIAG.C

	Copyright (C) 1999 Datalight, Inc.
	All Rights Reserved

	Diagnostic testing Sockets/DOS using CAPI

	$DOCHISTORY:
	1.00 1999-06-07 First version
	1.10 1999-07-02 Sockets 1.1 release
	1.11 1999-08-13 Added test for extended register save
	1.12 1999-12-15 Changed test for Bad Request
	1.13 2000-01-05 Test blocking write of 0 bytes
	1.14 2000-04-12 Conditional compile to show GetNetInfo results

	Compile and link with CAPI.C
	Use TABS=3 spaces for properly formatted source

	CAPIDIAG requires a remote server with both a UDP and TCP echo server
	for a complete test. NT server, Unix or Inet can provide this. It can
	also be run without such a server as a stand-alone test using only the
	internal loopback capabilities of Sockets.
	Extended register test must be enabled by defining EXTREG_TEST. It has
	only been tested on BC3.1.
*/

#ifdef EXTREG_TEST
#pragma inline
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <conio.h>
#include <process.h>
#include <stdarg.h>
#include <time.h>
#include <dos.h>
#include <ctype.h>
#include "compiler.h"
#include "capi.h"
#include "capidiag.pt"

#ifndef enable
	#define enable _enable
#endif
#ifndef disable
	#define disable _disable
#endif

int iClientSock, iServerSock;
NET_ADDR sClientAddr = {0,7,0,0}, sServerAddr = {0,0,7,0};

int iFailAction;		// action on failure
int iFail;				// fail/pass
int iVerbose;			// display tests
WORD wTimeOut = 5;	// timeout value
WORD wTimeOutHalf = 2;
int iFailures;
int iNumberTests;

char szData[] = "0123456789abcdefghijklmnopqrstuvwxyzABC";
char rgcReadBuf[80];
DWORD dwMyIPAddress;
DWORD dwServerIPAddress;
BYTE bDebugServer;		// debug the server
char szDiagnostic[400];	// Currently running diagnostic

// Client event variables, set at interrupt time
volatile int iEventPosted, iEventError;    //client callback related

typedef struct sClientEventStruc
{
	int iEventCount;		// count of events
	int iEventError;		// last event error
} CLIENT_EVENTS;

volatile CLIENT_EVENTS arsClientEvents[MAX_AS_EVENT + 2];

char *ppszEventMessage[] =
{
	"NET_AS_ALARM",
	"NET_AS_OPEN",
	"NET_AS_RCV",
	"NET_AS_XMT",
	"NET_AS_XMT_FLUSH",
	"NET_AS_FCLOSE",
	"NET_AS_CLOSE",
	"NET_AS_ERROR",
	"NET_AS_UNKNOWN",
};

int main(int argc, char *argv[])
{
	WORD wLen;
	char szNameBuf[80];
	BYTE pbBuf[16];
	char *pszArg;
	int i;
	KERNEL_CONFIG sKernelConfig;

	while (argc > 1 && argv[1][0] == '/')
	{
		pszArg = argv[1] + ((argv[1][2] == '=') ? 3 : 2);
		switch (tolower(argv[1][1]))
		{
			case 'd':
				bDebugServer = 1;
				break;
			default:
				printf(	"CAPIDIAG [/d][/l=<level>][/v=<verbose>] [<server>]\n"
							"/d - Debug server\n"
							"/f=<fail_action>\n"
							"   0 print error and continue\n"
							"   1 Print error and wait for Esc\n"
							"   2 Print error and exit\n"
							"/t=<timeout_seconds>\n"
							"/v=<verbosity> add values below\n"
							"   0 only print errors\n"
							"   1 print names of all tests\n"
							"   2 print additional information on errors\n"
							"   4 print summary of events\n"
							"   8 print kernel configuration\n");
				exit(0);
			case 'f':
				iFailAction = atoi(pszArg);
				break;
			case 't':
				wTimeOut = atoi(pszArg);
				if ((wTimeOutHalf = wTimeOut / 2) == 0)
					wTimeOutHalf = 1;
				break;
			case 'v':
				iVerbose = atoi(pszArg);
				break;
		}
		--argc;
		++argv;
	}
	wLen = sizeof(szNameBuf);
	szNameBuf[0] = 0;
	i = GetKernelInformation(0, K_INF_HOSTNAME, 0, szNameBuf, &wLen);
	printf("CAPI diagnostic 1.14 on '%s'\n", szNameBuf);
	Start("GetKernelConfig");
	if (GetKernelConfig(&sKernelConfig) < 0)
		Fail(0);
	End();
	if (iVerbose & 8)
	{
		printf("Kernel configuration:\n"
			"Number of TCP connections allowed       %5d\n"
			"Number of UDP connections allowed       %5d\n"
			"Number of IP connections allowed        %5d\n"
			"Number of RAW_NET connections allowed   %5d\n"
			"Number of TCP connections in use        %5d\n"
			"Number of UDP connections in use        %5d\n"
			"Number of IP connections in use         %5d\n"
			"Number of RAW_NET connections in use    %5d\n"
			"Number of active Dos Compatible Sockets %5d\n"
			"Number of active Sockets                %5d\n"
			"Maximum header on an attached network   %5d\n"
			"Maximum trailer on an attached network  %5d\n"
			"Size of a large packet buffer           %5d\n"
			"Number of network interfaces attached   %5d\n"
			"Milliseconds since kernel started       %5ld\n"
			"IP broadcast address in use   %s\n\n",
				sKernelConfig.bKMaxTcp,sKernelConfig.bKMaxUdp,
				sKernelConfig.bKMaxIp,sKernelConfig.bKMaxRaw,
				sKernelConfig.bKActTcp,sKernelConfig.bKActUdp,
				sKernelConfig.bKActIp,sKernelConfig.bKActRaw,
				sKernelConfig.wKActDCS,sKernelConfig.wKActSoc,
				sKernelConfig.bKMaxLnh,sKernelConfig.bKMaxLnt,
				sKernelConfig.bKLBUF_SIZE,sKernelConfig.bKNnet,
				sKernelConfig.dwKCticks,
				IPAddressString(sKernelConfig.dwKBroadcast));
	}
	Start("GetKernelInformation(K_INF_HOSTNAME)");
	if (i < 0)
		Fail(0);
	End();

	wLen = 16;
	Start("GetKernelInformation(K_INF_DNS_SERVERS)");
	if (GetKernelInformation(0, K_INF_DNS_SERVERS, 0, pbBuf, &wLen) < 0)
		Fail(0);
	else if (iVerbose & 8)
	{
		printf("DNS(%d): ",wLen);
		for (i = 0;i < wLen;i += 4)
			printf("%s ",IPAddressString(*(DWORD *)(pbBuf + i)));
		printf("\n");
	}
	End();

	wLen = 6;
	Start("GetKernelInformation(K_INF_MAC_ADDR)");
	if (GetKernelInformation(0, K_INF_MAC_ADDR, 0, pbBuf, &wLen) < 0)
		Fail(0);
	else if (iVerbose & 8)
	{
		printf("MAC address(%d): ",wLen);
		for (i = 0;i < wLen;i++)
			printf("%02X ",pbBuf[i]);
		printf("\n");
	}
	End();

	wLen = 4;
	Start("GetKernelInformation(K_INF_IP_ADDRESS)");
	if (GetKernelInformation(0, K_INF_IP_ADDR, 0, (char *)&dwMyIPAddress, &wLen) < 0)
		Fail(0);
	End();
	printf("Local IP address:  %s\n",IPAddressString(dwMyIPAddress));
	dwServerIPAddress = dwMyIPAddress;
	RunDiagnostics();
	if (argc > 1)
	{
		Start("ResolveName(\"%s\")",argv[1]);
		if ((dwServerIPAddress = ResolveName(argv[1], szNameBuf, sizeof(szNameBuf))) == 0)
			Fail(0);
		End();
		RunDiagnostics();
	}
	printf("Number of tests: %d, Failures: %d\n", iNumberTests, iFailures);
	return 0;
}

void RunDiagnostics(void)
{
	int i;
	char szAddress[40];
	DWORD dwAddress;

	printf("Remote IP address: %s\n", IPAddressString(dwServerIPAddress));
	sClientAddr.dwRemoteHost = dwServerIPAddress;

	sprintf(szAddress,"%s",IPAddressString(dwServerIPAddress));
	Start("ParseAddress");
	if ((dwAddress = ParseAddress(szAddress)) == 0L)
		Fail(0);
	else if (dwAddress != dwServerIPAddress)
		Fail("&returned %s", IPAddressString(dwAddress));
	End();

	// Check for bad system call returning an error
	Start("Bad request");
	if (BadRequest() != -1 || iNetErrNo != ERR_BAD_SYS_CALL)
		Fail(0);
	End();

	Start("GetVersion");
	if ((i = GetVersion()) < 0)
		Fail(0);
	else if (i != 0x214)
		Fail("& returned %#x, expected 0x214",i);
	End();

	Start("ICMPPing(10)");
	if (ICMPPing(dwServerIPAddress,10) < 0)
	{
		Fail(0);
		End();
	}
	else
	{
		End();
		Start("ICMPPing(1000)");
		if (ICMPPing(dwServerIPAddress,1000) < 0)
			Fail(0);
		End();
	}
	// flush keyboard buffer
	while (kbhit())
		getch();

	TestSocket(STREAM);
	TestSocket(DATA_GRAM);
}

#ifdef EXTREG_TEST
struct sExStruct
{
	DWORD eax;
	DWORD ebx;
	DWORD ecx;
	DWORD edx;
} sExs, sExr;
#endif

int TestSocket(int iMode)
{
	DWORD dwLocalIPAddress;
	NET_ADDR sPeerAddr;
	NET_INFO sNetInfo;
	int iRead, iWritten, iSize, iResult;
	char *pszMode = iMode == STREAM ? "STREAM" : "DATA_GRAM";
	long lIflags = 0L, lOflags = 0L, lMask;

#ifdef EXTREG_TEST
	// save Extended registers
	asm	.386;
	asm	mov dword ptr sExs.eax,eax;
	asm	mov dword ptr sExs.ebx,ebx;
	asm	mov dword ptr sExs.ecx,ecx;
	asm	mov dword ptr sExs.edx,edx;
	asm	.8086
#endif

	if (dwServerIPAddress == dwMyIPAddress)
		StartListen(iMode);
	sClientAddr.wRemotePort = 7;
	sClientAddr.wLocalPort = 0;
	iEventPosted = 0;
	memset((void *)arsClientEvents,0,sizeof(arsClientEvents));

	Start("GetSocket(%s)", pszMode);
	if ((iClientSock = GetSocket()) < 0)
		Fail(0);
	End();

	// make iClientSock non-blocking
	Start("%s SetSocketOption(NET_OPT_NON_BLOCKING)", pszMode);
	if (SetSocketOption(iClientSock, 0, NET_OPT_NON_BLOCKING, 1, 1) < 0)
		Fail(0);
	End();

	// set invalid ASYNC Handler
	Start("%s SetAsyncNotification(INVALID)",pszMode);
	if (SetAsyncNotification(
						iClientSock, 20, Handler, 0) != (int far *)-1 ||
						iNetErrNo != ERR_BAD_ARG)
		Fail(0);
	End();
	// set ASYNC Handler on OPEN
	Start("%s SetAsyncNotification(OPEN)",pszMode);
	if (SetAsyncNotification(iClientSock, NET_AS_OPEN, Handler, 0) == (int far *)-1)
		Fail(0);
	End();
	// set ASYNC Handler on RCV
	Start("%s SetAsyncNotification(RCV)",pszMode);
	if (SetAsyncNotification(iClientSock, NET_AS_RCV, Handler, 0) == (int far *)-1)
		Fail(0);
	End();
	// set ASYNC Handler on XMT
	Start("%s SetAsyncNotification(XMT)",pszMode);
	if (SetAsyncNotification(iClientSock, NET_AS_XMT, Handler, 0) == (int far *)-1)
		Fail(0);
	End();
	// set ASYNC Handler on CLOSE
	Start("%s SetAsyncNotification(CLOSE)",pszMode);
	if (SetAsyncNotification(iClientSock, NET_AS_CLOSE, Handler, 0) == (int far *)-1)
		Fail(0);
	End();
	// set ASYNC Handler on FCLOSE
	Start("%s SetAsyncNotification(FCLOSE)",pszMode);
	if (SetAsyncNotification(iClientSock, NET_AS_FCLOSE, Handler, 0) == (int far *)-1)
		Fail(0);
	End();
	// set ASYNC Handler on ERROR
	Start("%s SetAsyncNotification(ERROR)",pszMode);
	if (SetAsyncNotification(iClientSock, NET_AS_ERROR, Handler, 0) == (int far *)-1)
		Fail(0);
	End();

	// test SetAlarm
	Start("SetAlarm");
	if (SetAlarm(iClientSock, 200, Handler, 0) < 0)
		Fail(0);
	End();
	Start("Alarm event");
	if (arsClientEvents[NET_AS_ALARM].iEventCount != 0)
		Fail("&fired too soon");
	SetTimer(wTimeOut);
	while (!iEventPosted)
	{
		if (DiagWait())
		{
			iFail = 1;
			break;
		}
		if (arsClientEvents[NET_AS_ALARM].iEventCount != 0)
			break;
	}
	End();

	iEventPosted = 0;
	Start("ConnectSocket(%s)",pszMode);
	if (ConnectSocket(iClientSock, iMode, &sClientAddr) < 0)
		Fail(0);
	End();

	if (iMode == STREAM)
	{
		Start("%s open event",pszMode);
		SetTimer(wTimeOut);
		while (!iEventPosted)
		{
			if (DiagWait())
			{
				iFail = 1;
				goto EventAbort;
			}
		}
		switch (iEventPosted)
		{
		case NET_AS_ERROR:
			Fail("&NET_AS_ERROR %s", GetErrorString(iEventError));
			break;
		case NET_AS_CLOSE:
			Fail("&NET_AS_CLOSE");
			break;
		case NET_AS_OPEN:
			// This is the "good" case
			if (iVerbose & 1)
				printf("Connection established ");
			break;
		default:
			Fail("&Unexpected event %d",pszMode, iEventPosted);
			break;
		}
		EventAbort:
		End();
	}

	// Check for no receive events
	Start("%s no receive event",pszMode);
	if ((iResult = arsClientEvents[NET_AS_RCV].iEventCount) != 0)
		Fail("&count is %d, expected 0", iResult);
	End();

	// Check for no transmit events
	Start("%s no transmit event",pszMode);
	if ((iResult = arsClientEvents[NET_AS_XMT].iEventCount) != 0)
		Fail("&count is %d, expected 0", iResult);
	End();

	Start("%s GetAddress",pszMode);
	if ((dwLocalIPAddress = GetAddress(iClientSock)) == 0)
		Fail(0);
	if (dwLocalIPAddress != dwMyIPAddress)
		Fail("&returned %s", IPAddressString(dwLocalIPAddress));
	End();

	Start("%s GetPeerAddress",pszMode);
	if (GetPeerAddress(iClientSock,&sPeerAddr) < 0)
		Fail(0);
	if (sPeerAddr.dwRemoteHost != dwServerIPAddress)
		Fail("&returned wrong RemoteHost %s",IPAddressString(sPeerAddr.dwRemoteHost));
	if (sPeerAddr.wRemotePort != 7)
		Fail("&returned wrong RemotePort %d",sPeerAddr.wRemotePort);
	End();

	Start("%s GetNetInfo",pszMode);
	if (GetNetInfo(iClientSock,&sNetInfo) < 0)
		Fail(0);
	if (sNetInfo.dwIpAddress != dwMyIPAddress)
		Fail("&returned wrong IPAddress %s",IPAddressString(sNetInfo.dwIpAddress));
#ifdef SHOW_NETINFO
	printf("NetInfo():\ndwIpAddress %s\n",IPAddressString(sNetInfo.dwIpAddress));
	printf("dwIpSubnet  %s\n",IPAddressString(sNetInfo.dwIpSubnet));
	printf("iUp %d,iLanLen %d\nLanAddr %02x:%02x:%02x:%02x:%02x:%02x\n",
		sNetInfo.iUp,sNetInfo.iLanLen,
		sNetInfo.pLanAddr[0],
		sNetInfo.pLanAddr[1],
		sNetInfo.pLanAddr[2],
		sNetInfo.pLanAddr[3],
		sNetInfo.pLanAddr[4],
		sNetInfo.pLanAddr[5]);
#endif
	End();

	// Nothing should be in receive queue
	Start("%s ReadSocket(buf,NET_FLG_PEEK | NET_FLG_NON_BLOCKING)", pszMode);
	if ((iRead = ReadSocket(iClientSock, rgcReadBuf,sizeof(rgcReadBuf),0,NET_FLG_PEEK | NET_FLG_NON_BLOCKING)) != -1 ||
			iNetErrNo != ERR_WOULD_BLOCK)
		Fail("&unexpected count %d",iRead);
	End();
	Start("%s ReadSocket(0,NET_FLG_PEEK)",pszMode);
	if ((iRead = ReadSocket(iClientSock, 0,sizeof(rgcReadBuf),0,NET_FLG_PEEK)) != -1 ||
			iNetErrNo != ERR_WOULD_BLOCK)
		Fail("&unexpected count %d",iRead);
	End();
	Start("%s ReadSocket(0,0)",pszMode);
	if ((iRead = ReadSocket(iClientSock, rgcReadBuf,sizeof(rgcReadBuf),0,0)) != -1 ||
			iNetErrNo != ERR_WOULD_BLOCK)
		Fail("&unexpected count %d",iRead);
	End();

	Start("DisableAsyncNotification from enabled");
	if ((iResult = DisableAsyncNotification()) < 0)
		Fail(0);
	if (iResult != 1)
		Fail("&result %d, expected 1", iResult);
	End();

	Start("DisableAsyncNotification from disabled");
	if ((iResult = DisableAsyncNotification()) < 0)
		Fail(0);
	if (iResult != 0)
		Fail("&result %d, expected 0", iResult);
	End();

	// Send some data to echo server
	Start("%s WriteSocket",pszMode);
	if ((iWritten = WriteSocket(iClientSock, szData, sizeof(szData), 0)) < 0)
		Fail(0);
	else if (iWritten != sizeof(szData))
		Fail("&short write: %d expected %d",iWritten,sizeof(szData));
	End();

	delay(500);		// give time to receive response

	// Check for no receive events
	Start("%s no receive event",pszMode);
	if ((iResult = arsClientEvents[NET_AS_RCV].iEventCount) != 0)
		Fail("&count is %d, expected 0", iResult);
	End();

	// Check for no transmit events
	Start("%s no transmit event",pszMode);
	if ((iResult = arsClientEvents[NET_AS_XMT].iEventCount) != 0)
		Fail("&count is %d, expected 0", iResult);
	End();

	Start("EnableAsyncNotification from disabled");
	if ((iResult = EnableAsyncNotification()) < 0)
		Fail(0);
	if (iResult != 0)
		Fail("&result %d, expected 0", iResult);
	End();

	Start("EnableAsyncNotification from enabled");
	if ((iResult = EnableAsyncNotification()) < 0)
		Fail(0);
	if (iResult != 1)
		Fail("&result %d, expected 1", iResult);
	End();

	if (iMode == DATA_GRAM)
	{
		// Just check datagram count, no reading should take place
		Start("%s ReadSocket(pBuf,NET_FLG_PEEK|NET_FLG_NON_BLOCKING)==1", pszMode);
		memset(rgcReadBuf,0,sizeof(rgcReadBuf));
		if ((iRead = ReadSocketCount(iClientSock,rgcReadBuf,sizeof(rgcReadBuf),
								1,NET_FLG_PEEK | NET_FLG_NON_BLOCKING)) == 1)
		{
			if (memcmp(szData,rgcReadBuf,iRead) == 0)
				Fail("Readback should NOT have worked:\n%.*s\n%.*s",iRead,szData,iRead,rgcReadBuf);
		}
		End();
	}

	// Just check count, no reading should take place
	Start("%s ReadSocket(0,NET_FLG_PEEK)==%d", pszMode, sizeof(szData));
	memset(rgcReadBuf,0,sizeof(rgcReadBuf));
	if ((iRead = ReadSocketCount(iClientSock, 0, sizeof(rgcReadBuf),
									sizeof(szData),NET_FLG_PEEK)) == sizeof(szData))
	{
		if (memcmp(szData,rgcReadBuf,iRead) == 0)
			Fail("Readback should NOT have worked:\n%.*s\n%.*s",iRead,szData,iRead,rgcReadBuf);
	}
	End();

	// Check for receive upcall having happened
	Start("%s receive event",pszMode);
	if (arsClientEvents[NET_AS_RCV].iEventCount == 0)
		Fail(0);
	End();

	// Send some more data to echo server
	Start("%s WriteSocket",pszMode);
	if ((iWritten = WriteSocket(iClientSock, szData, sizeof(szData), 0)) < 0)
		Fail(0);
	else if (iWritten != sizeof(szData))
		Fail("&short write: %d expected %d",iWritten,sizeof(szData));
	End();

	if (iMode == DATA_GRAM)
	{
		// Just check datagram count, no reading should take place
		Start("%s ReadSocket(pBuf,NET_FLG_PEEK|NET_FLG_NON_BLOCKING)==2", pszMode);
		memset(rgcReadBuf,0,sizeof(rgcReadBuf));
		if ((iRead = ReadSocketCount(iClientSock,rgcReadBuf,sizeof(rgcReadBuf),
								2,NET_FLG_PEEK | NET_FLG_NON_BLOCKING)) == 2)
		{
			if (memcmp(szData,rgcReadBuf,iRead) == 0)
				Fail("Readback should NOT have worked:\n%.*s\n%.*s",iRead,szData,iRead,rgcReadBuf);
		}
		End();
	}

	// Just check count, no reading should take place
	// PCTCP seems to remember the count once it has been read and not
	// update it until it has been read. The following delay is used to
	// make sure the first read is OK
	delay(100);		// PCTCP needs it
	iSize = iMode == STREAM ? 2 * sizeof(szData) : sizeof(szData);
	Start("%s ReadSocket(0,NET_FLG_PEEK)==%d", pszMode, iSize);
	memset(rgcReadBuf,0,sizeof(rgcReadBuf));
	if ((iRead = ReadSocketCount(iClientSock, 0, sizeof(rgcReadBuf),
							iSize,NET_FLG_PEEK)) == iSize)
	{
		if (memcmp(szData,rgcReadBuf,iRead) == 0)
			Fail("Readback should NOT have worked:\n%.*s\n%.*s",iRead,szData,iRead,rgcReadBuf);
	}
	End();

	// Read back without de-queueing data
	Start("%s ReadSocket(pBuf,NET_FLG_PEEK)==%d", pszMode, sizeof(szData));
	memset(rgcReadBuf,0,sizeof(rgcReadBuf));
	if ((iRead = ReadSocketCount(iClientSock, rgcReadBuf,sizeof(szData),sizeof(szData),NET_FLG_PEEK)) == sizeof(szData))
	{
		if (memcmp(szData,rgcReadBuf,iRead) != 0)
			Fail("Readback error:\n%.*s\n%.*s",iRead,szData,iRead,rgcReadBuf);
	}
	End();

	if (iMode == DATA_GRAM)
	{
		// Just check datagram count, no reading should take place
		Start("%s ReadSocket(pBuf,NET_FLG_PEEK|NET_FLG_NON_BLOCKING)==2", pszMode);
		memset(rgcReadBuf,0,sizeof(rgcReadBuf));
		if ((iRead = ReadSocketCount(iClientSock,rgcReadBuf,sizeof(rgcReadBuf),
								2,NET_FLG_PEEK | NET_FLG_NON_BLOCKING)) == 2)
		{
			if (memcmp(szData,rgcReadBuf,iRead) == 0)
				Fail("Readback should NOT have worked:\n%.*s\n%.*s",iRead,szData,iRead,rgcReadBuf);
		}
		End();
	}

	// Just check count, no reading should take place
	Start("%s ReadSocket(0,NET_FLG_PEEK)==%d", pszMode, iSize);
	memset(rgcReadBuf,0,sizeof(rgcReadBuf));
	if ((iRead = ReadSocketCount(iClientSock, 0,
								sizeof(rgcReadBuf), iSize,NET_FLG_PEEK)) == iSize)
	{
		if (memcmp(szData,rgcReadBuf,iRead) == 0)
			Fail("Readback should NOT have worked:\n%.*s\n%.*s",iRead,szData,iRead,rgcReadBuf);
	}
	End();

	// Read data
	Start("%s ReadSocket(pBuf,0)==%d", pszMode, sizeof(szData));
	memset(rgcReadBuf,0,sizeof(rgcReadBuf));
	if ((iRead = ReadSocketCount(iClientSock, rgcReadBuf,sizeof(szData),
												sizeof(szData),0)) == sizeof(szData))
	{
		if (memcmp(szData,rgcReadBuf,iRead) != 0)
			Fail("Readback error:\n%.*s\n%.*s",iRead,szData,iRead,rgcReadBuf);
	}
	End();

	if (iMode == DATA_GRAM)
	{
		// Just check datagram count, no reading should take place
		Start("%s ReadSocket(pBuf,NET_FLG_PEEK|NET_FLG_NON_BLOCKING)==1", pszMode);
		memset(rgcReadBuf,0,sizeof(rgcReadBuf));
		if ((iRead = ReadSocketCount(iClientSock,rgcReadBuf,sizeof(rgcReadBuf),
								1,NET_FLG_PEEK | NET_FLG_NON_BLOCKING)) == 1)
		{
			if (memcmp(szData,rgcReadBuf,iRead) == 0)
				Fail("Readback should NOT have worked:\n%.*s\n%.*s",iRead,szData,iRead,rgcReadBuf);
		}
		End();
	}

	// debug code
	exit (0);
	// end debug

	// Just check count, no reading should take place
	Start("%s ReadSocket(0,NET_FLG_PEEK)==%d", pszMode, sizeof(szData));
	memset(rgcReadBuf,0,sizeof(rgcReadBuf));
	if ((iRead = ReadSocketCount(iClientSock, 0, sizeof(rgcReadBuf),
									sizeof(szData),NET_FLG_PEEK)) == sizeof(szData))
	{
		if (memcmp(szData,rgcReadBuf,iRead) == 0)
			Fail("Readback should NOT have worked:\n%.*s\n%.*s",iRead,szData,iRead,rgcReadBuf);
	}
	End();

	// Read data
	Start("%s ReadSocket(pBuf,0)==%d", pszMode, sizeof(szData));
	memset(rgcReadBuf,0,sizeof(rgcReadBuf));
	if ((iRead = ReadSocketCount(iClientSock, rgcReadBuf,sizeof(rgcReadBuf),
												sizeof(szData),0)) == sizeof(szData))
	{
		if (memcmp(szData,rgcReadBuf,iRead) != 0)
			Fail("Readback error:\n%.*s\n%.*s",iRead,szData,iRead,rgcReadBuf);
	}
	End();

#ifdef FORCE
	// Force an error
	Start("%s ReadSocket(pBuf,0)", pszMode);
	memset(rgcReadBuf,0,sizeof(rgcReadBuf));
	if ((iRead = ReadSocketCount(iClientSock, rgcReadBuf,sizeof(rgcReadBuf),sizeof(szData),0)) == sizeof(szData))
	{
		if (memcmp(szData,rgcReadBuf,iRead) != 0)
			Fail("Readback error:\n%.*s\n%.*s",iRead,szData,iRead,rgcReadBuf);
	}
	End();
#endif

	// next read should block
	Start("%s ReadSocket(pBuf,0)", pszMode);
	if ((iRead = ReadSocket(iClientSock, rgcReadBuf,sizeof(rgcReadBuf),0,0)) != -1)
		Fail("&unexpected data count %d", iRead);
	else if (iNetErrNo != ERR_WOULD_BLOCK)
		Fail(0);
	End();

	// Check for receive events
	Start("%s receive event",pszMode);
	if ((iResult = arsClientEvents[NET_AS_RCV].iEventCount) == 0)
		Fail("&count is 0, something expected");
	End();

	if (iMode == STREAM)
	{
		// Check for transmit events
		Start("%s transmit event",pszMode);
		if ((iResult = arsClientEvents[NET_AS_XMT].iEventCount) == 0)
			Fail("&count is 0, something expected");
		End();
	}

	if (iMode == STREAM)
	{
		iEventPosted = 0;
		Start("EofSocket()");
		if (EofSocket(iClientSock) < 0)
			Fail(0);
		End();
		Start("%s close event",pszMode);
		SetTimer(wTimeOut);
		while (!iEventPosted)
		{
			if (arsClientEvents[NET_AS_FCLOSE].iEventCount &&
								arsClientEvents[NET_AS_CLOSE].iEventCount)
				break;
			if (DiagWait())
			{
				iFail = 1;
				break;
			}
		}
		End();
	}


	// contrary to documentation the next test should pass
	Start("IsSocket on normal socket");
	if (IsSocket(iClientSock) < 0)
		Fail(0);
	End();

	Start("%s ReleaseSocket(client)",pszMode);
	if (ReleaseSocket(iClientSock) < 0)
		Fail(0);
	End();

	Start("IsSocket on released normal socket");
	if (IsSocket(iClientSock) != -1 || iNetErrNo != ERR_NOT_NET_CONN)
		Fail(0);
	End();

	if (dwServerIPAddress == dwMyIPAddress)
	{
		Start("%s ReleaseSocket(server)",pszMode);
		if (ReleaseSocket(iServerSock) < 0)
			Fail(0);
		End();
		StartListen(iMode);
	}

	// test Dos Compatible sockets

	Start("GetDCSocket");
	if ((iClientSock = GetDCSocket()) < 0)
		Fail(0);
	End();

	Start("IsSocket on 0");
	if (IsSocket(0) != -1 || iNetErrNo != ERR_NOT_NET_CONN)
		Fail(0);
	End();

	Start("IsSocket");
	if (IsSocket(iClientSock) < 0)
		Fail(0);
	End();

	// test SelectSocket
	lMask = 1 << iClientSock;
	Start("SelectSocket");
	if (SelectSocket(32,&lIflags,&lOflags) != -1 || iNetErrNo != ERR_BAD_ARG)
		Fail(0);
	End();

	lIflags = lOflags = 0L;
	Start("SelectSocket");
	if (SelectSocket(31,&lIflags,&lOflags) < 0)
		Fail(0);
	else if (lIflags || lOflags)
		Fail("&returned lIflags %#lx lOflags %#lx (expected %#lx %#lx)",
			lIflags,lOflags,0l,0l);
	End();

	// make iClientSock non-blocking
	Start("%s SetSocketOption(NET_OPT_NON_BLOCKING)", pszMode);
	if (SetSocketOption(iClientSock, 0, NET_OPT_NON_BLOCKING, 1, 1) < 0)
		Fail(0);
	End();

	// make iClientSock wait for flush
	Start("%s SetSocketOption(NET_OPT_WAIT_FLUSH)", pszMode);
	if (SetSocketOption(iClientSock, 0, NET_OPT_WAIT_FLUSH, 1, 1) < 0)
		Fail(0);
	End();

	sClientAddr.wRemotePort = 7;
	sClientAddr.wLocalPort = 0;
	Start("ConnectSocket(%s)",pszMode);
	if (ConnectSocket(iClientSock, iMode, &sClientAddr) < 0)
		Fail(0);
	End();

	SetTimer(wTimeOut);
	lOflags = 0L;
	Start("%s SelectSocket output", pszMode);
	while (1)
	{
		if (DiagWait())
		{
			iFail = 1;
			break;
		}
		if (SelectSocket(31,&lIflags,&lOflags) < 0)
		{
			Fail(0);
			break;
		}
		if (lOflags & lMask)
			break;
	}
	if (lIflags & lMask)
		Fail("&returned lIflags %#lx lOflags %#lx (expected %#lx %#lx)",
			lIflags,lOflags,0l,lMask);
	End();

	// Send some data to echo server
	Start("%s DC WriteSocket",pszMode);
	if ((iWritten = WriteSocket(iClientSock, szData, sizeof(szData), 0)) < 0)
		Fail(0);
	else if (iWritten != sizeof(szData))
		Fail("&short write: %d expected %d",iWritten,sizeof(szData));
	End();

	if (iMode == STREAM)
	{
		Start("Flush timeout");
		SetTimer(wTimeOutHalf);
		while (1)
		{
			if (DiagWait)
				break;
			if (SelectSocket(31,&lIflags,&lOflags) < 0)
			{
				Fail(0);
				break;
			}
			if (lIflags & lMask)
			{
				Fail(0);
				break;
			}
		}
		End();
		Start("FlushSocket");
		if (FlushSocket(iClientSock) < 0)
			Fail(0);
		End();
	}
	SetTimer(wTimeOut);
	Start("%s SelectSocket input", pszMode);
	while (1)
	{
		if (DiagWait())
		{
			iFail = 1;
			break;
		}
		if (SelectSocket(31,&lIflags,&lOflags) < 0)
		{
			Fail(0);
			break;
		}
		if (lIflags & lMask)
			break;
	}
	End();

	// a read should succeed
	Start("%s DC ReadSocket(0,NET_FLG_PEEK)",pszMode);
	if ((iRead = ReadSocket(iClientSock, 0, sizeof(rgcReadBuf),0,
											NET_FLG_PEEK)) <= 0)
		Fail(0);
	End();

	Start("%s DC ReadSocket(pBuf,0)==%d", pszMode, sizeof(szData));
	memset(rgcReadBuf,0,sizeof(rgcReadBuf));
	if ((iRead = ReadSocketCount(iClientSock, rgcReadBuf,sizeof(szData),
												sizeof(szData),0)) == sizeof(szData))
	{
		if (memcmp(szData,rgcReadBuf,iRead) != 0)
			Fail("Readback error:\n%.*s\n%.*s",iRead,szData,iRead,rgcReadBuf);
	}
	End();

	Start("%s AbortSocket",pszMode);
	if (AbortSocket(iClientSock) < 0)
		Fail(0);
	End();

	if (dwServerIPAddress == dwMyIPAddress)
	{
		Start("%s ReleaseSocket(server)",pszMode);
		if (ReleaseSocket(iServerSock) < 0)
			Fail(0);
		End();
		StartListen(iMode);
	}

	// make a new connection with -1 which will be blocking
	sClientAddr.wRemotePort = 7;
	sClientAddr.wLocalPort = 0;
	Start("ConnectSocket(%s)",pszMode);
	if ((iClientSock = ConnectSocket(-1, iMode, &sClientAddr)) < 0)
		Fail(0);
	End();

	// Write zero bytes
	Start("%s WriteSocket(0,blocking)",pszMode);
	if ((iWritten = WriteSocket(iClientSock, szData, 0, 0)) < 0)
		Fail(0);
	else if (iWritten != 0)
		Fail("&return: %d, expected %d",iWritten,0);
	End();


	// Send some data to echo server
	Start("%s WriteSocket(blocking)",pszMode);
	if ((iWritten = WriteSocket(iClientSock, szData, sizeof(szData), 0)) < 0)
		Fail(0);
	else if (iWritten != sizeof(szData))
		Fail("&short write: %d expected %d",iWritten,sizeof(szData));
	End();

	Start("%s DC ReadSocket(pBuf,0)==%d", pszMode, sizeof(szData));
	memset(rgcReadBuf,0,sizeof(rgcReadBuf));
	if ((iRead = ReadSocketCount(iClientSock, rgcReadBuf,sizeof(szData),
												sizeof(szData),0)) == sizeof(szData))
	{
		if (memcmp(szData,rgcReadBuf,iRead) != 0)
			Fail("Readback error:\n%.*s\n%.*s",iRead,szData,iRead,rgcReadBuf);
	}
	End();

	Start("%s AbortDCSockets",pszMode);
	if (AbortDCSockets() < 0)
		Fail(0);
	End();

	Start("%s DC ReleaseSocket after AbortSocket",pszMode);
	if (ReleaseSocket(iClientSock) != -1 || iNetErrNo != ERR_NOT_NET_CONN)
		Fail(0);
	End();

	// test ReleaseDCSockets
	Start("GetDCSocket");
	if ((iClientSock = GetDCSocket()) < 0)
		Fail(0);
	End();

	Start("%s ReleaseDCSockets",pszMode);
	if (ReleaseDCSockets() < 0)
		Fail(0);
	End();

	Start("ReleaseSocket after ReleaseDCSockets");
	if (ReleaseSocket(iClientSock) != -1 || iNetErrNo != ERR_NOT_NET_CONN)
		Fail(0);
	End();

	// test ConvertDCSocket
	Start("GetDCSocket");
	if ((iClientSock = GetDCSocket()) < 0)
		Fail(0);
	End();

	Start("ConvertDCSocket");
	if ((iResult = ConvertDCSocket(iClientSock)) < 0)
		Fail(0);
	else if (iResult == iClientSock || iResult < 64)
		Fail("&%d converted to %d", iClientSock, iResult);
	End();

	Start("ReleaseSocket on converted from socket");
	if (ReleaseSocket(iClientSock) != -1 || iNetErrNo != ERR_NOT_NET_CONN)
		Fail(0);
	End();

	Start("ReleaseSocket on converted to socket");
	if (ReleaseSocket(iResult) < 0)
		Fail(0);
	End();

	Start("ReleaseSocket on converted from socket");
	if (ReleaseSocket(iClientSock) != -1 || iNetErrNo != ERR_NOT_NET_CONN)
		Fail(0);
	End();

	if (dwServerIPAddress == dwMyIPAddress)
	{
		Start("%s ReleaseSocket(server)",pszMode);
		if (ReleaseSocket(iServerSock) < 0)
			Fail(0);
		End();
	}

#ifdef EXTREG_TEST
	// load Extended registers
	asm	.386;
	asm	mov dword ptr sExr.eax,eax;
	asm	mov dword ptr sExr.ebx,ebx;
	asm	mov dword ptr sExr.ecx,ecx;
	asm	mov dword ptr sExr.edx,edx;
	asm	.8086

	Start("Extended register save");
	iNetErrNo = 0;		// prevent confusing result
	if (((sExr.eax ^ sExs.eax) & 0xffff0000L) ||
		 ((sExr.ebx ^ sExs.ebx) & 0xffff0000L) ||
		 ((sExr.ecx ^ sExs.ecx) & 0xffff0000L) ||
		 ((sExr.edx ^ sExs.edx) & 0xffff0000L))
		Fail(0);
	End();
#endif

	// print event statistics if requested
	if (iVerbose & 4)
	{
		int i;

		printf("%s events:\n",pszMode);
		for (i = 0;i <= MAX_AS_EVENT + 1;i++)
		{
			printf("%-16.16s %4d %s\n",ppszEventMessage[i],
							arsClientEvents[i].iEventCount,
							i == NET_AS_ERROR ?
							GetErrorString(arsClientEvents[i].iEventError) : "");
		}
	}
	return 0;
}

// Read expected number of bytes or until interrupted
int ReadSocketCount(int iSocket,char *pBuf,int iBufLen,int iExpectedLen,
	WORD wFlags)
{
	int i, iRead = 0;
	char *pReadBuf = pBuf;
	SetTimer(wTimeOut);
	while (1)
	{
		if (DiagWait())
		{
			iFail = 1;
			if (iVerbose & 2)
				printf("Read %d\n",iRead);
			return -1;
		}
		if ((i = ReadSocket(iSocket,pReadBuf,iBufLen - iRead,0,wFlags)) < 0)
		{
			if (iNetErrNo != ERR_WOULD_BLOCK)
			{
				Fail(0);
				return -1;
			}
		}
		else if (i == 0) {
			Fail("Peer closed connection while reading %d byte(s)",
							iExpectedLen);
			return -1;
		}
		else {
			if (wFlags & NET_FLG_PEEK)
				iRead = i;
			else
				iRead += i;
			if (iRead >= iExpectedLen)
				break;
			if (pReadBuf && !(wFlags & NET_FLG_PEEK))
				pReadBuf += i;
		}
	}
	if (iRead != iExpectedLen)
		Fail("&unexpected count %d, expected %d",iRead,iExpectedLen);
	return iRead;
}

/*
 Async notification Handler for client socket - called by Sockets.
 Do NOT use any functions making DOS calls, eg printf().
 */

int _loadds far Handler(void)
{
	// count the event
	__asm	{
		mov iEventPosted, cx;
		mov iEventError, dx;
	}
	// stay within limits of array
	if ((WORD)iEventPosted > MAX_AS_EVENT + 1)
		iEventPosted = MAX_AS_EVENT + 1;
	arsClientEvents[iEventPosted].iEventCount++;
	arsClientEvents[iEventPosted].iEventError = iEventError;

#ifdef EXTREG_TEST
	// damage the extended registers to test saving
	asm .386
	asm add eax,010000h;
	asm add ebx,020000h;
	asm add ecx,040000h;
	asm add edx,080000h;
	asm .8086
#endif

	return 0;
}

// Async notification Handler for server - called by Sockets.
// Do NOT use any functions making DOS calls, eg printf().

int iServerMode;

enum		// Server action event logging
{
	SA_ENTER,
	SA_READ,
	SA_WRITE,
	SA_WRITETO,
	SA_EOF,
	SA_FULL
};

char *ppszActionMessage[] =
{
	"Enter",
	"ReadSocket",
	"WriteSocket",
	"WriteToSocket",
	"EofSocket",
	"Queue full",
};

typedef struct sEventQue
{
	int iAction;
	int iEvent;
	DWORD dwArg;
	int iError;
} EVENT_QUE;

#define EVENT_QUE_SIZE 40

EVENT_QUE arsEventQue[EVENT_QUE_SIZE];

int iEventQueIn;		// event queue input index
int iEventQueOut;		// event queue output index
int iEventQueCount;	// count of events

void QueEvent(int iAction,int iEvent,DWORD dwArg,int iError)
{
	EVENT_QUE *psEventQue;

	if (iEventQueCount == EVENT_QUE_SIZE - 1)
	{
		arsEventQue[iEventQueIn].iAction = SA_FULL;
		return;
	}
	iEventQueCount++;
	psEventQue = arsEventQue + iEventQueIn;
	psEventQue->iAction = iAction;
	psEventQue->iEvent = iEvent;
	psEventQue->dwArg = dwArg;
	psEventQue->iError = iError;
	if (++iEventQueIn == EVENT_QUE_SIZE)
   	iEventQueIn = 0;
}

int far ServerHandler(int iSocket, int iEvent, DWORD dwArg)
{
	char rgcBuf[200];
	int iCount;
	NET_ADDR sAddr;

	QueEvent(SA_ENTER,iEvent,dwArg,0);
	switch (iEvent)
	{
	case NET_AS_ERROR:
	case NET_AS_CLOSE:
		break;
	case NET_AS_OPEN:
		break;
	case NET_AS_FCLOSE:		// peer closed connection
	case NET_AS_RCV:
		//only echo data back
		if ((iCount = ReadSocket(iSocket, rgcBuf, sizeof(rgcBuf), &sAddr, 0)) > 0)
		{
			if (iServerMode == STREAM)
			{
				if (WriteSocket(iSocket, rgcBuf, iCount, 0) < 0)
					QueEvent(SA_WRITE,iEvent,dwArg,iNetErrNo);
			}
			else
			{
				if (WriteToSocket(iSocket, rgcBuf, iCount, &sAddr, 0) < 0)
					QueEvent(SA_WRITETO,iEvent,dwArg,iNetErrNo);
			}
		}
		else
				QueEvent(SA_READ,iEvent,dwArg,iNetErrNo);
		if (iEvent == NET_AS_FCLOSE)
			if (EofSocket(iSocket) < 0)
				QueEvent(SA_EOF,iEvent,dwArg,iNetErrNo);
		break;
	case NET_AS_XMT:
		break;
	}
	return 0;
}

/*
Start a server socket listening on the port number contained in the
sServerAddr structure.
Set all the api callbacks concerning this socket to ServerHandler.

Returns:
	The newly created server socket.
*/
int StartListen(int iMode)
{
	int retv = 0;
	char *pszMode = iMode == STREAM ? "STREAM" : "DATA_GRAM";

	iServerMode = iMode;
	Start("%s server GetSocket",pszMode);
	if ((iServerSock = GetSocket()) < 0)
	{
		Fail(0);
		retv = -1;
	}
	End();
	// make iServerSock non-blocking
	Start("%s server SetSockOption(NET_OPT_NON_BLOCKING)",pszMode);
	if (SetSocketOption(iServerSock, 0, NET_OPT_NON_BLOCKING, 1, 1) < 0)
	{
		Fail(0);
		retv = -2;
	}
	End();

	// set ASYNC Handler on OPEN
	Start("%s server SetAsyncNotification(NET_AS_OPEN)",pszMode);
	if (SetAsyncNotification(iServerSock, NET_AS_OPEN, AsyncNotificationHandler, (DWORD)ServerHandler) == (int far *)-1)
	{
		Fail(0);
		retv = -4;
	}
	End();
	// set ASYNC Handler on CLOSE
	Start("%s server SetAsyncNotification(NET_AS_CLOSE)",pszMode);
	if (SetAsyncNotification(iServerSock, NET_AS_CLOSE, AsyncNotificationHandler, (DWORD)ServerHandler) == (int far *)-1)
	{
		Fail(0);
		retv = -5;
	}
	End();
	// set ASYNC Handler on FCLOSE
	Start("%s server SetAsyncNotification(NET_AS_FCLOSE)",pszMode);
	if (SetAsyncNotification(iServerSock, NET_AS_FCLOSE, AsyncNotificationHandler, (DWORD)ServerHandler) == (int far *)-1)
	{
		Fail(0);
		retv = -6;
	}
	End();
	// set ASYNC Handler on ERROR
	Start("%s server SetAsyncNotification(NET_AS_ERROR)",pszMode);
	if (SetAsyncNotification(iServerSock, NET_AS_ERROR, AsyncNotificationHandler, (DWORD)ServerHandler) == (int far *)-1)
	{
		Fail(0);
		retv = -7;
	}
	End();
	// set ASYNC Handler on RECEIVE
	Start("%s server SetAsyncNotification(NET_AS_RCV)",pszMode);
	if (SetAsyncNotification(iServerSock, NET_AS_RCV, AsyncNotificationHandler, (DWORD)ServerHandler) == (int far *)-1)
	{
		Fail(0);
		retv = -8;
	}
	End();
	// set ASYNC Handler on XMIT
	Start("%s server SetAsyncNotification(NET_AS_XMT)",pszMode);
	if (SetAsyncNotification(iServerSock, NET_AS_XMT, AsyncNotificationHandler, (DWORD)ServerHandler) == (int far *)-1)
	{
		Fail(0);
		retv = -9;
	}
	End();
	Start("%s server ListenSocket",pszMode);
	sServerAddr.wRemotePort = 0;
	sServerAddr.dwRemoteHost = 0;
	sServerAddr.wLocalPort = 7;
	if (ListenSocket(iServerSock, iMode, &sServerAddr) < 0)
	{
		Fail(0);
		return -3;
	}
	End();
	return retv;
}

/*
Create a human understandable string from a Sockets error code.

Argument:
	uErrCode - The sockets error code.

Returns:
	A pointer to the (static) string representation of the error.
*/
char *GetErrorString(unsigned uErrCode)
{
	static char rgcUnk[30];
	static char *rgszErrs[] =
	{
		"NO_ERR",					// 0	// No error
		"ERR_IN_USE",				// 1	// A connection already exists
		"ERR_DOS",					// 2	// A DOS error occured
		"ERR_NO_MEM",				// 3	// No memory to perform function
		"ERR_NOT_NET_CONN",		// 4	// Connection does not exist
		"ERR_ILLEGAL_OP",			// 5	// Protocol or mode not supported
		"ERR_BAD_PACKET",
		"ERR_NO_HOST",				// 7	// No host address specified
		"ERR_CANT_OPEN",
		"ERR_NET_UNREACHABLE",
		"ERR_HOST_UNREACHABLE",
		"ERR_PROTOCOL_UNREACHABLE",
		"ERR_PORT_UNREACHABLE",
		"ERR_TIMEOUT",				// 13	// The function timed out
		"ERR_HOST_UNKNOWN",		// 14	// Unknown host has been specified
		"ERR_NO_SERVERS",
		"ERR_SERVER_ERR",
		"ERR_BAD_FORMAT",
		"ERR_BAD_ARG",				// 18	// Bad arguments
		"ERR_EOF",					// 19	// The connection has been closed by peer
		"ERR_RESET",				// 20	// The connection has been reset by peer
		"ERR_WOULD_BLOCK",		// 21	// Operation would block
		"ERR_UNBOUND",				// 22	// The descriptor has not been assigned yet
		"ERR_NO_SOCKET",			// 23	// No descriptor is available
		"ERR_BAD_SYS_CALL",		// 24	// Bad parameter in call
		"ERR_CANT_BROADCAST",
		"ERR_NOT_ESTAB",			// 26	// The connection has not been established
		"ERR_RE_ENTRY",			// 27	// The kernel is in use, try again later
	};

	if (uErrCode == ERR_API_NOT_LOADED)
		return "Sockets API not loaded";
	if ((uErrCode & 0xff) > ERR_RE_ENTRY)
	{
		sprintf(rgcUnk, "Unknown error 0x%04X", uErrCode);
		return rgcUnk;
	}
	return rgszErrs[uErrCode & 0xff];
}

char *IPAddressString(DWORD dwIPAddress)
{
	static char sz[20];

	sprintf(sz, "%u.%u.%u.%u",
			((BYTE *)&dwIPAddress)[0],((BYTE *)&dwIPAddress)[1],
			((BYTE *)&dwIPAddress)[2],((BYTE *)&dwIPAddress)[3]);
	return sz;
}

time_t dwTargetTime;

// Set the timer for a wait
void SetTimer(WORD wTimeOut)
{
	dwTargetTime = time(0) + wTimeOut;
}

// Wait for keyboard or timeout
int DiagWait(void)
{
	char *pszArgvl[2];	//used in spawning COMSPEC
	WORD wKey;

	if (bDebugServer)
	{
		while (disable(),iEventQueCount)
		{
			EVENT_QUE *psEventQue = arsEventQue + iEventQueOut;
			int iAction = psEventQue->iAction;
			int iEvent = psEventQue->iEvent;
			DWORD dwArg = psEventQue->dwArg;
			int iError = psEventQue->iError;

			--iEventQueCount;
			if (++iEventQueOut == EVENT_QUE_SIZE)
				iEventQueOut = 0;
			enable();
			printf("Server %s %s arg %lu %s\n",
				ppszActionMessage[iAction],
				ppszEventMessage[iEvent],dwArg,GetErrorString(iError));
		}
		enable();
	}
	if (kbhit())
	{
		wKey = getch();
		if (wKey == 0)
			/* function key */
			wKey = 1000 + getch();
		switch (wKey)
		{
			case 1045:		/* alt X */
				ReleaseSocket(iClientSock);
				ReleaseSocket(iServerSock);
				printf("User exit\n");
				exit(2);
				break;
			case 1035:	// Alt H
				printf( "\n"
						"Escape        Abort test\n"
						"Alt-E         Execute command.com\n"
						"Alt-H         Help\n"
						"Alt-X         eXit\n"
						"Any other key Display current test\n");
				break;
			case 1018:	// alt e
				if ((pszArgvl[0] = getenv("COMSPEC")) == 0)
					pszArgvl[0] = "\\COMMAND.COM";
				pszArgvl[1] = 0;
				spawnvp(P_WAIT, pszArgvl[0], pszArgvl);
				break;
			default:
				if (wKey == 0x1b)	// Escape
				{
					Fail("&- user abort");
					return 1;
				}
				printf("Waiting for %s\n", szDiagnostic);
		}
	}
	if (dwTargetTime && (dwTargetTime < time(0)))
	{
		Fail("&- timeout");
		return 1;
	}
	return 0;
}

// Start a test
// Save test name for diagnostics

int Start(char *szFormat, ...)
{
	va_list ap;

	iFail = 0;
	iNumberTests++;
	va_start(ap, szFormat);
	vsprintf(szDiagnostic, szFormat, ap);
	va_end(ap);
	if (iVerbose & 1)
		printf("%s ",szDiagnostic);
	return 0;
}

// fail a test
int Fail (char *szFormat, ...)
{
	va_list ap;
	char szLine[400];

	if (iFail)
		return 0;	// fail once only
	++iFailures;
	iFail = 1;
	if (!(iVerbose & 1))
		printf("%s ",szDiagnostic);
	if (szFormat)
	{
		va_start(ap, szFormat);
		vsprintf(szLine, szFormat + (*szFormat == '&' ? 1 : 0), ap);
		va_end(ap);
		if (*szFormat == '&')
			printf("%s - failed %s\n", szLine, GetErrorString(iNetErrNo));
		else
			printf("- failed %s\n%s\n", GetErrorString(iNetErrNo),szLine);
	}
	else
		printf("- failed %s\n", GetErrorString(iNetErrNo));
	if (iFailAction)
	{
		if (iFailAction < 2)
		{
			dwTargetTime = 0;		// don't time out
			printf("Press Esc to continue\n");
			while (!DiagWait());
			iFail = 0;
		}
		else
			exit(1);
	}
	return 0;
}

// End a test. Print message if enabled
void End(void)
{
	if ((iVerbose & 1) && !iFail)
	{
		printf("- Passed\n");
	}
}

int BadRequest(void)
{
	union REGS r;
   extern int iSocketsApiInterrupt;

	r.h.ah = 0x70;
	int86(iSocketsApiInterrupt,&r,&r);
	if (r.x.flags & 1) {
		iNetErrNo = r.h.al;
		iNetSubErrNo = r.h.ah;
		return -1;
	}
	return r.x.ax;
}

